import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Camera, Star, BookOpen, User, Calendar } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const [courses, setCourses] = useState([])

  useEffect(() => {
    loadCourses()
    const interval = setInterval(loadCourses, 500)
    return () => clearInterval(interval)
  }, [])

  const loadCourses = () => {
    const saved = localStorage.getItem('photoCourses')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setCourses(parsed)
      } catch (e) {
        setCourses([])
      }
    } else {
      const defaultCourses = [
        {
          id: 1,
          module: 'Основы композиции',
          instructor: 'Анна Петрова',
          completionDate: '2025-03-25',
          rating: 5
        },
        {
          id: 2,
          module: 'Работа со светом',
          instructor: 'Михаил Соколов',
          completionDate: '2025-03-20',
          rating: 5
        },
        {
          id: 3,
          module: 'Портретная съемка',
          instructor: 'Елена Волкова',
          completionDate: '2025-03-15',
          rating: 4
        }
      ]
      setCourses(defaultCourses)
      localStorage.setItem('photoCourses', JSON.stringify(defaultCourses))
    }
  }

  const totalCourses = courses.length
  const uniqueInstructors = new Set(courses.map(c => c.instructor)).size
  const averageRating = courses.length > 0 
    ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1)
    : 0

  return (
    <div className="my-courses">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <h1 className="hero-title">Мои курсы</h1>
          <p className="hero-description">
            Ваш прогресс в обучении цифровой фотографии
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.1 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ x: 3, transition: { duration: 0.2 } }}
          >
            <BookOpen size={32} strokeWidth={2} color="#2563EB" />
            <div className="stat-content">
              <div className="stat-value">{totalCourses}</div>
              <div className="stat-label">Курсов</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ x: 3, transition: { duration: 0.2 } }}
          >
            <Star size={32} strokeWidth={2} color="#2563EB" />
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ x: 3, transition: { duration: 0.2 } }}
          >
            <User size={32} strokeWidth={2} color="#2563EB" />
            <div className="stat-content">
              <div className="stat-value">{uniqueInstructors}</div>
              <div className="stat-label">Преподавателей</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/course-1.jpg',
          '/images/course-2.jpg',
          '/images/course-3.jpg',
          '/images/course-4.jpg'
        ]} />

        <motion.section 
          className="courses-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <h2 className="section-title">Ваши курсы</h2>
          {courses.length > 0 ? (
            <div className="courses-list">
              {courses.map((course, index) => (
                <motion.div 
                  key={course.id} 
                  className="course-item"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 3, transition: { duration: 0.2 } }}
                >
                  <div className="course-main">
                    <div className="course-info">
                      <h3 className="course-module">{course.module}</h3>
                      <div className="course-details">
                        <div className="course-detail">
                          <User size={18} strokeWidth={2} />
                          <span>{course.instructor}</span>
                        </div>
                        <div className="course-detail">
                          <Calendar size={18} strokeWidth={2} />
                          <span>
                            {new Date(course.completionDate).toLocaleDateString('ru-RU', { 
                              day: 'numeric', 
                              month: 'long', 
                              year: 'numeric' 
                            })}
                          </span>
                        </div>
                      </div>
                    </div>
                    <div className="course-rating">
                      {Array.from({ length: 5 }, (_, i) => (
                        <Star
                          key={i}
                          size={20}
                          fill={i < course.rating ? '#2563EB' : 'none'}
                          stroke={i < course.rating ? '#2563EB' : '#CBD5E1'}
                          strokeWidth={2}
                        />
                      ))}
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <Camera size={64} strokeWidth={1.5} />
              <h3 className="empty-title">Нет курсов</h3>
              <p className="empty-text">Запишитесь на курс прямо сейчас</p>
            </div>
          )}
        </motion.section>
      </div>
    </div>
  )
}

export default MyCourses

